#!/bin/sh -u
#-------------------------------------------------------------------------------
# birthday.sh                                    (C) 2003 T.Birnthaler OSTC GmbH
#-------------------------------------------------------------------------------
# TODO: Schalter -f = Feiertage (festive days) einbauen (auch bewegliche!)
#       Schalter -w = weekday einbauen
#       Schalter -t = today einbauen
#       Schalter -o = only future einbauen
#-------------------------------------------------------------------------------
# Geburtstage aus der Datei "~/.birthday" mit dem heutigen Datum vergleichen
# und passende Tage innerhalb eines Bereiches (whlbar, Std: 5 Tage) um den
# heutigen Tag anzeigen. Gltige Eintrge bestehen aus einer Zeile mit mind.
# 3 Worten in folgendem Format:
#
#   MONAT TAG TEXT... [P]
#
#   MONAT = Jan Feb Mar Apr May Jun Jul Aug Sep Oct Nov Dec (Englisch!)
#   TAG   = 1..31
#
# + Leerraum ist in einer Zeile beliebig verwendbar.
# + Kommentarzeilen mit "#" beliebig verwendbar (nur Zeile fr sich!)
# + Leerzeilen und Trennlinien "---" sind erlaubt.
# + Sortierung ist beliebig (nach Datum sortiert => Ausgabe bersichtlicher).
# + Private Eintrge (mit Wort "P" am Ende gekennz.) sind getrennt ausgebbar.
# + Das Geburtsjahr ist im TEXT in Klammern "(NNNN)" (4 Stellen!) angebbar und
#   wird mit dem Schalter "-a" (age) in das Alter "[MM]" umgerechnet.
# + Testeintrge enthalten das Wort 'TEST' in beliebiger Schreibweise.
#-------------------------------------------------------------------------------
# Beispiel fr Inhalt von "~/.birthday":
# Jan 1  Tom Cat
# Jan 3  Foo Bar P
# Jan 4  Linus Torvalds (1974)
# ... .. ...
# Dec 24 Weihnachten
# Dec 31 Sylvester
#-------------------------------------------------------------------------------
# TIP: Um beim Start eines Terminal-Fensters die Geburtstags-Informationen nur
# im ersten gestarteten Terminal und nur als normaler Benutzer (nicht als root)
# auszugeben, die folgende Zeile in "~/.bashrc" bzw. "~/.profile" aufnehmen:
#
#	[ $(tty) = "/dev/pts/1" -o $(tty) = "/dev/tty1" ] && \
#		[ $UID -ne 0 ] && \
#		/PATH/TO/birthday.sh
#-------------------------------------------------------------------------------
# TRICK: Um negative Untergrenze des Bereiches zu umgehen bzw. Bereiche die ber
# die linke/rechte Jahresgrenze hinausreichen: Datum um 1 Jahr = 365 Tage in die
# Zukunft schieben und die beiden angrenzenden Jahre zustzlich betrachten.
#-------------------------------------------------------------------------------

# Konstanten
BIRTHDAYFILE=$HOME/.birthday      # Geburtstagsliste ("Jan 15 Tom Cat... P")
       DELTA=5                    # Anzuzeigender Bereich um heutiges Datum
        LANG=en                   # wg. Mar/Mae, May/Mai, Oct/Okt, Dec/Dez
       TODAY=$(date +"%b %d %Y")  # Heute
        MARK="***"                # Kennzeichen fr aktuelles Datum
    YEAR2AGE=                     # Schalter -a (year2age) aus
       DEBUG=                     # Schalter -d (debug) aus
     PRIVATE=                     # Schalter -p (private) aus
        TEST=                     # Schalter -t (test) aus

# Usage-Meldung
Usage()
{
	echo "Zeigt alle Eintrge der Datei '~/.birthday' (Format: MONAT TAG TEXT...[P]) an,"
	echo "die ein Datum NN Tage (Std: $DELTA) um den heutigen Tag ($TODAY) herum haben"
	echo "bzw. sucht die Eintrge heraus, die den Text 'NAME...' enthalten."
	echo
	[ "$1" ] && echo "error: $1"
	echo "usage: $(basename $0) [OPTIONEN] [NAME...]"
	echo "       -a        Alter ausrechnen falls in der Form '(NNNN)' vorhanden [age]"
	echo "       -m TEXT   Kennzeichen fr heutiges Datum (Std: $MARK) [mark]"
	echo "       -r DELTA  Tages-Bereich um Datum herum (Std: $DELTA) [range]"
	echo "       -p        Nur private Eintrge anzeigen (Wort 'P' am Ende) [private]"
	echo
	echo "       -d        Debugmodus [debug]"
	echo "       -f        1. Tag im Jahr testen [first]"
	echo "       -l        Letzten Tag im Jahr testen [last]"
	echo "       -t DATUM  Eintrge mit Text 'TEST' zu DATUM 'MMM DD YYYY' anzeigen [test]"
	exit
}

# Geburtstagsdatei vorhanden?
[ -e $BIRTHDAYFILE ] || Usage "Geburtstagsdatei '$BIRTHDAYFILE' nicht vorhanden"

#-------------------------------------------------------------------------------
# Eintrag formatieren und ausgeben (Jahreszahl => Alter, ...)
# Parameter: $1=MONTH $2=DAY $3-$N=TEXT
#-------------------------------------------------------------------------------
PrintEntry()
{
	MONTH=$1; DAY=$2; shift 2; TEXT="$*"
# Einstellige Tage zweistellig machen (egal ob mit fhrender 0 oder nicht!)
	DAY=$(expr $DAY + 0)
	[ $DAY -le 9 ] && DAY=" $DAY"
# Jahreszahlen in runden Klammern in Alter umrechnen (eckige Klammern)
	if [ "$YEAR2AGE" ]
	then
		while echo "$TEXT" | grep '([12][09][0-9][0-9])' > /dev/null
		do
			YEAR=$(echo $TEXT | sed 's/^.*(\([12][09][0-9][0-9]\)).*$/\1/')
			AGE=$(expr $CURRYEAR - $YEAR)
			TEXT=$(echo "$TEXT" | sed "s/($YEAR)/[$AGE]/")
		done
	fi
	echo "$MONTH $DAY $TEXT"
}

# Optionen einlesen
while getopts ":adpfhlm:r:t:" OPT; do # ":" at start suppresses error message
	case $OPT in                      # ":" after letter means option with parameter
		a) YEAR2AGE=1 ;;
		d) DEBUG=1 ;;
		f) TODAY="Jan  1 $(date +%Y)" ;;
		h) Usage;;
		l) TODAY="Dec 31 $(date +%Y)" ;;
		m) MARK="$OPTARG" ;;
		r) DELTA="$OPTARG" ;;
		p) PRIVATE=1 ;;
		t) TEST=1; TODAY="$OPTARG" ;;
		*) Usage "Unbekannte Option -$OPTARG" ;;
	esac
done
shift $(expr $OPTIND - 1)

# Aktueller Tag und gleicher Tag Jahr davor bzw. Jahr danach
 CURRDAY=$(date -d "$TODAY" +%j)        # $TODAY wg. -f/-l/-t
 CURRDAY=$(expr 365 + $CURRDAY)         # Schaltjahr nicht bercksichtigen ???
FIRSTDAY=$(expr $CURRDAY - $DELTA)
 LASTDAY=$(expr $CURRDAY + $DELTA)

# Aktuelles Jahr und Jahr davor bzw. danach
CURRYEAR=$(date -d "$TODAY" +%Y)
PREVYEAR=$(expr $CURRYEAR - 1)
NEXTYEAR=$(expr $CURRYEAR + 1)

# Debugausgaben
[ "$DEBUG" ] && echo "# Delta: $DELTA"
[ "$DEBUG" ] && echo "# Tage:  $FIRSTDAY $CURRDAY $LASTDAY"
[ "$DEBUG" ] && echo "# Jahre: $PREVYEAR $CURRYEAR $NEXTYEAR"

#-------------------------------------------------------------------------------
# Nach Eintrgen mit angegebenen Namen suchen und dann Stop
#-------------------------------------------------------------------------------
if [ $# -gt 0 ]
then
	for NAME
	do
		grep -i "$NAME" $BIRTHDAYFILE |
		while read MONTH DAY TEXT
		do
			[ "$DEBUG" ] && echo -n "$MONTH $DAY $TEXT: "
			PrintEntry $MONTH $DAY "$TEXT"
		done
	done
	exit
fi

#-------------------------------------------------------------------------------
# Alle Eintrge abarbeiten und zeitlich passende aufbereitet ausgeben
#-------------------------------------------------------------------------------
LENGTH=$(cat $BIRTHDAYFILE |
         grep "[a-zA-Z]" | grep -iv "TEST" | grep -v "^#" |
         wc -l | tr -d " ")
echo "berprfe Geburtstagsliste... ($LENGTH Eintrge)"

while read MONTH DAY TEXT
do
	[ "$MONTH" = "#" ] && continue # Kommentar ignorieren
	[ -z "$DAY" ]  && continue     # Falsches Format ignorieren
# Nur private Eintrge? => Andere ignorieren
	[ "$PRIVATE" ] && echo "$TEXT" | grep -iv ' P *$' > /dev/null && continue
# Test-Eintrge oder normale Eintrge anzeigen? => Jeweils andere ignorieren
	[ "$TEST" ]    && echo "$TEXT" | grep -iv 'TEST'  > /dev/null  && continue
	[ -z "$TEST" ] && echo "$TEXT" | grep -i  'TEST'  > /dev/null  && continue
	[ "$DEBUG" ]   && echo -n "$MONTH $DAY $TEXT: "
# Geburtstag mit Bereich um heutiges Datum vergleichen
	for YEAR in $CURRYEAR $PREVYEAR $NEXTYEAR             # Reihenfolge = Speed!
	do
		BIRTH=$(date -d "$MONTH $DAY $YEAR" +%j)
		[ $YEAR -eq $CURRYEAR ] && BIRTH=$(expr 365 + $BIRTH)
		[ $YEAR -eq $NEXTYEAR ] && BIRTH=$(expr 2 \* 365 + $BIRTH)
		[ "$DEBUG" ] && echo -n "$FIRSTDAY <= $BIRTH <= $LASTDAY: "
# Geburtstag im gesuchten Bereich um heutiges Datum? => Ausgeben
		if [ $BIRTH -ge $FIRSTDAY -a $BIRTH -le $LASTDAY ]
		then
# Heutiges Datum ? => Markieren
			[ $BIRTH -eq $CURRDAY ] && TEXT="$TEXT $MARK"
			PrintEntry $MONTH $DAY "$TEXT"
			continue 2                                    # Speed!
		fi
	done
	[ "$DEBUG" ] && echo
done < $BIRTHDAYFILE
